#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmMaintainComments;

#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{

	[TestFixture]
	public class BOL_CannedComment : BaseTest 
	{

		
		private DataTable _dtCannedComment;
		private BOL.CannedComment _tstCannedComment;
		private Guid _commentGuid;

		[SetUp]	
		protected void SetUp() 
		{				
			this.BaseSetUp();

			if (this.RefreshData)
			{
				//CannedComments are created on UnitTest Setup and should exist
				this._dtCannedComment = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM CannedComment WHERE RecordStatusCode = 'A' AND CannedCommentText <> 'Other' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'",  true);
				
				this._commentGuid = (Guid)this._dtCannedComment.Rows[0][TABLES.CannedComment.CannedCommentGuid];

				this.RefreshData = false;
			}

			this._tstCannedComment = new BOL.CannedComment(this._commentGuid);
			this._tstCannedComment.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.Default);
		}

		[Test]
		public void Ctor_DataRow_Pass()
		{			
			BOL.CannedComment tmpComment = new BOL.CannedComment(this._dtCannedComment.Rows[0]);

			Assert.IsTrue(tmpComment.CannedCommentGuid == this._commentGuid);
			Assert.IsTrue(tmpComment.CannedCommentType == this._dtCannedComment.Rows[0][TABLES.CannedComment.CannedCommentCategoryCode].ToString());
		}

		[Test]
		public void Ctor_DataRow_Fail()
		{
			DataTable dt = new DataTable();
			dt.Columns.Add(TABLES.CannedComment.CannedCommentGuid,typeof(Guid));
			dt.Columns.Add(TABLES.CannedComment.SortColumn,typeof(Int16));
			//
			DataRow dr = dt.NewRow();
			//
			BOL.CannedComment cannedComment = new BOL.CannedComment(dr);
			//
			Assert.AreEqual(Guid.Empty,cannedComment.CannedCommentGuid);
			Assert.AreEqual(0,cannedComment.SortIndex);
		}

		[Test]
		public void Ctor_Guid_Pass()
		{			
			BOL.CannedComment tmpComment = new BOL.CannedComment(this._commentGuid);

			Assert.IsTrue(tmpComment.CannedCommentGuid == this._commentGuid);
			Assert.IsTrue(tmpComment.IsNew == false);
		}

		[Test]
		public void Ctor_Guid_Fail()
		{			
			Guid unitGuid = Guid.NewGuid();
			BOL.CannedComment tmpComment = new BOL.CannedComment(unitGuid);
			
			Assert.IsTrue(tmpComment.CannedCommentGuid == unitGuid);
			Assert.IsTrue(tmpComment.IsNew == true);
		}

		[Test]
		public void Comments_Pass()
		{
			string tstValue = System.DateTime.Now.ToString();

			this._tstCannedComment.Comments = tstValue;
			Assert.IsTrue(this._tstCannedComment.Comments == tstValue);
		}

		[Test]
		public void Comments_Fail1()
		{			
			this._tstCannedComment.Comments = string.Empty;
			Assert.IsTrue(this._tstCannedComment.IsRuleBroken(BROKEN.Comments));
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void Comments_Fail2()
		{			
			this._tstCannedComment.Comments = "".PadRight(256,'X');
		}

		[Test]
		public void CannedCommentType_Pass()
		{			
			string tstValue = System.DateTime.Now.ToString();
			this._tstCannedComment.CannedCommentType = tstValue;

			Assert.IsTrue(this._tstCannedComment.CannedCommentType == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void CannedCommentType_Fail()
		{			
		}

		[Test]
		public void RecordStatusCode_Pass()
		{			
			this._tstCannedComment.RecordStatusCode = Common.RecordStatusCode.ErrorCorrected;

			Assert.IsTrue(this._tstCannedComment.RecordStatusCode == Common.RecordStatusCode.ErrorCorrected);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RecordStatusCode_Fail()
		{			
		}


		[Test]
		public void SortIndex_Pass()
		{			
			this._tstCannedComment.SortIndex = short.MaxValue;

			Assert.IsTrue(this._tstCannedComment.SortIndex == short.MaxValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SortIndex_Fail()
		{			
		}

		[Test]
		public void CannedCommentGuid_Pass()
		{			
			Guid tstValue = Guid.NewGuid();
			this._tstCannedComment.CannedCommentGuid = tstValue;

			Assert.IsTrue(this._tstCannedComment.CannedCommentGuid == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void CannedCommentGuid_Fail()
		{			
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{	
			DataRow dr = this._tstCannedComment.LoadDataRowFromThis(this._dtCannedComment.NewRow());

			Assert.IsTrue(dr[TABLES.CannedComment.CannedCommentGuid].ToString() == this._tstCannedComment.CannedCommentGuid.ToString());
			Assert.IsTrue(dr[TABLES.CannedComment.CannedCommentText].ToString() == this._tstCannedComment.Comments);
		}		
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{	
		}

		[Test]
		public void GetCannedComments_Pass()
		{
			DataTable dtTest = BOL.CannedComment.GetCannedComments();

			Assert.IsTrue(dtTest.Rows.Count >= 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetCannedComments_Fail()
		{
		}

		[Test]
		public void GetCannedComments_String_Pass()
		{
			string categoryCode = this._dtCannedComment.Rows[0][TABLES.CannedComment.CannedCommentCategoryCode].ToString();
			DataTable dtTest = BOL.CannedComment.GetCannedComments(categoryCode);

			Assert.IsTrue(dtTest.Rows.Count > 0);
		}

		[Test]
		public void GetCannedComments_String_Fail()
		{
			string categoryCode = "@@";
			DataTable dtTest = BOL.CannedComment.GetCannedComments(categoryCode);

			Assert.IsTrue(dtTest.Rows.Count == 0);
		}

		[Test]
		public void GetCannedComments_String_String_Pass()
		{
			string categoryCode = this._dtCannedComment.Rows[0][TABLES.CannedComment.CannedCommentCategoryCode].ToString();
			DataTable dtTest = BOL.CannedComment.GetCannedComments(UnitTestConfig.TestDivisionCode, categoryCode);

			Assert.IsTrue(dtTest.Rows.Count > 0);
		}

		[Test]
		public void GetCannedComments_String_String_Fail()
		{
			string categoryCode = "@@";
			DataTable dtTest = BOL.CannedComment.GetCannedComments("XXX", categoryCode);

			Assert.IsTrue(dtTest.Rows.Count == 0);
		}

		[Test]
		public void GetCannedCommentList_Pass()
		{	
			string categoryCode = this._dtCannedComment.Rows[0][TABLES.CannedComment.CannedCommentCategoryCode].ToString();

			System.Collections.ArrayList alTest = BOL.CannedComment.GetCannedCommentsList(UnitTestConfig.TestDivisionCode, categoryCode);

			Assert.IsTrue(alTest.Count > 0);
		}

		[Test]
		public void GetCannedCommentList_Fail()
		{	
			string categoryCode = "@@";
			
			System.Collections.ArrayList alTest = BOL.CannedComment.GetCannedCommentsList("XXX", categoryCode);

			Assert.IsTrue(alTest.Count == 0);
		}

		
		[Test]
		public void GetCannedCommentCategories_Pass()
		{	
			DataTable dtTest = BOL.CannedComment.GetCannedCommentCategories();

			Assert.IsTrue(dtTest.Rows.Count > 0);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetCannedCommentCategories_Fail()
		{	
		}


		[Test]
		public void Save_Pass()
		{	
			this.RefreshData = true;

			byte[] commentRV = (byte[])this._dtCannedComment.Rows[0][TABLES.CannedComment.RowVersion];

			DataTable dtUComment = SprocHelper.GetUpdateCannedCommentSprocTable(this._commentGuid, commentRV);
			//
			//Adding columns for insert too - for 100% coverage
			dtUComment.Columns.Add(TABLES.CannedComment.LastUpdateDate, typeof(DateTime));
			dtUComment.Columns.Add(TABLES.CannedComment.CannedCommentCategoryCode, typeof(string));
			dtUComment.Columns.Add(TABLES.CannedComment.CannedCommentText, typeof(string));
			dtUComment.Columns.Add(TABLES.CannedComment.DivisionCode, typeof(string));
			dtUComment.AcceptChanges();
			//
			//Update data row to allow save
			dtUComment.Rows[0][TABLES.CannedComment.LastUpdateDate] = DateTime.Now;
			//
			//Now create a new comment for 100% coverage
			Guid commentGuid = Guid.NewGuid();
			DataRow drNewComment = dtUComment.NewRow();
			drNewComment[TABLES.CannedComment.CannedCommentGuid] = commentGuid;
			drNewComment[TABLES.CannedComment.CannedCommentCategoryCode] = "SG";
			drNewComment[TABLES.CannedComment.CannedCommentText] = "Unit Test 867";
			drNewComment[TABLES.CannedComment.SortColumn] = 999;
			drNewComment[TABLES.CannedComment.DivisionCode] = UnitTestConfig.TestDivisionCode;
			drNewComment[TABLES.CannedComment.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drNewComment[TABLES.CannedComment.LastUpdateFunctionId] = Common.UpdateFunction.UnitTests;
			dtUComment.Rows.Add(drNewComment);
			//
			bool retValue = BOL.CannedComment.Save(dtUComment, Common.UpdateFunction.UnitTests);
			Assert.IsTrue(retValue);
			//
			if (retValue)
			{
				//Cleanup: Delete the comment we just created.
				UnitTestUtil.RunSQL("DELETE FROM CannedComment WHERE CannedCommentGuid = '"+commentGuid.ToString()+"'");
			}
		}

		[Test]
		[ExpectedException(typeof(Common.RowVersionException))]
		public void Save_Fail()
		{	
			byte[] commentRV = new byte[8];

			DataTable dtUComment = SprocHelper.GetUpdateCannedCommentSprocTable(this._commentGuid, commentRV);
			dtUComment.AcceptChanges();
			dtUComment.Rows[0][TABLES.CannedComment.RowVersion] = new byte[8];


			bool retValue = BOL.CannedComment.Save(dtUComment, Common.UpdateFunction.UnitTests);
		}


		[Test]
		public void CombinedComment_Pass()
		{	
			string stuff = System.DateTime.Now.ToString();
			//
			string tstValue = BOL.CannedComment.CombinedComment(this._commentGuid, stuff);
			string combined = this._tstCannedComment.Comments;

			//Normal flow
			if (!combined.EndsWith(".")) combined += ".";
			combined += " " + stuff;
			Assert.AreEqual(combined, tstValue, "1");
			
			//Other comment
			DataRow drData = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM CannedComment WHERE CannedCommentText = 'Other'");
			tstValue = BOL.CannedComment.CombinedComment((Guid)drData[TABLES.CannedComment.CannedCommentGuid],stuff);
			Assert.AreEqual(stuff, tstValue,"2");
			
			//Blank freetext
			tstValue = BOL.CannedComment.CombinedComment(this._commentGuid, "");
			Assert.AreEqual(this._tstCannedComment.Comments, tstValue, "3");

			//Test with a period at the end of the comment
			this._tstCannedComment.Comments += ".";
			tstValue = BOL.CannedComment.CombinedComment(this._commentGuid, stuff);
			combined = this._tstCannedComment.Comments;

			if (!combined.EndsWith(".")) combined += ".";
			combined += " " + stuff;
			Assert.AreEqual(combined, tstValue, "5");

			this.RefreshData = true;
		}
		
		[Test]
		public void CombinedComment_Fail()
		{
			string tstValue = BOL.CannedComment.CombinedComment(Guid.NewGuid(),"");
			Assert.AreEqual("",tstValue);
		}


	}
}
#endif


